/* libunwind - a platform-independent unwind library

This file is part of libunwind.

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and/or sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be
included in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.  */

#include "os-l4.h"
#include <limits.h>
#include <stdio.h>

#include "libunwind_i.h"
#include "os-l4.h"
#include <l4/re/c/rm.h>
#include <errno.h>
#include <sys/types.h>

int
l4_find_elf (l4_cap_idx_t rm, l4_addr_t addr, unsigned long *segbase,
             unsigned long *mapoff, char *path, size_t pathlen)
{
  l4_addr_t reg_addr;
  unsigned long size = 1;
  l4re_rm_offset_t off;
  l4re_rm_flags_t flags;
  l4re_ds_t m;

  reg_addr = addr;
  int r = l4re_rm_find_srv(rm, &reg_addr, &size, &off, &flags, &m);
  if (r < 0)
    {
      Debug(1, "Failed to find region for address 0x%lx.\n", addr);
      return r;
    }

  char name[100];
  l4re_rm_offset_t file_off;
  r = l4re_rm_get_info_srv(rm, reg_addr, name, sizeof(name), &file_off);
  if (r < 0 || !name[0])
    {
      Debug(1, "Region for address 0x%lx has no name.\n", addr);
      return r < 0 ? r : -1;
    }

  // TODO: segbase and mapoff should be aligned to page size (as regions are),
  //       however libunwind expects the mapoff to be exactly the same as in
  //       the elf file (dwarf_find_unwind_table()).
  *mapoff = file_off;
  // Regions in the region manager are always at least page size aligned.
  // Thus, when calculating the virtual load address of an ELF segment, we
  // have to borrow the lower bits from the file offset of that region.
  // TODO: Is this a valid assumption for ELF segments?
  *segbase = (reg_addr & L4_PAGEMASK) | (file_off & ~L4_PAGEMASK);
  strncpy(path, name, pathlen);
  path[pathlen - 1] = '\0';

  Debug(16,
        "ELF image for address 0x%lx: %s with segbase=0x%lx and mapoff=0x%lx\n",
        addr, name, *segbase, *mapoff);
  return 0;
}

int
tdep_get_elf_image (struct elf_image *ei, pid_t pid, unw_word_t ip,
                    unsigned long *segbase, unsigned long *mapoff,
                    char *path, size_t pathlen)
{
  // TODO: In local senario we could use dl_iterate_phdr instead.
  // reinterpret cast pid as capability to region manager...
  int r = l4_find_elf(pid_to_l4(pid), ip, segbase, mapoff, path, pathlen);
  if (r < 0)
    return r;

  return elf_map_image (ei, path);
}

pid_t
l4_to_pid(l4_cap_idx_t rm)
{
  assert(((l4_cap_idx_t)(pid_t)rm) == rm);
  return (pid_t)rm;
}

l4_cap_idx_t
pid_to_l4(pid_t pid)
{
  assert(((pid_t)(l4_cap_idx_t)pid) == pid);
  return (l4_cap_idx_t)pid;
}

pid_t
getpid(void)
{
  return l4_to_pid(l4re_global_env->rm);
}

#ifndef UNW_REMOTE_ONLY

void
tdep_get_exe_image_path (char *path)
{
  strcpy(path, program_invocation_name);
}

#endif /* !UNW_REMOTE_ONLY */
