/*
 * Copyright (C) 2017-2018, 2021, 2024 Kernkonzept GmbH.
 * Author(s): Alexander Warg <alexander.warg@kernkonzept.com>
 *
 * License: see LICENSE.spdx (in this directory or the directories above)
 */
#pragma once

#include <arm_hyp.h>
#include <l4/re/error_helper>

inline bool has_aarch32()
{
  l4_uint64_t aa64pfr0;
  asm ("mrs %0, ID_AA64PFR0_EL1" : "=r"(aa64pfr0));
  return (aa64pfr0 & 0x0f) == 2;
}

inline void arm_subarch_setup(void *vcpu, bool guest_64bit, bool pmsa)
{
  if (guest_64bit)
    {
      l4_umword_t hcr = l4_vcpu_e_read(vcpu, L4_VCPU_E_HCR);
      hcr |= 1UL << 18; // TID3: Trap ID Group 3 (feature system registers)
      hcr |= 1UL << 31; // set RW bit
      l4_vcpu_e_write(vcpu, L4_VCPU_E_HCR, hcr);
    }
  else if (!has_aarch32())
    L4Re::throw_error(-L4_ENOSYS, "CPU does not support 32-bit mode");

  unsigned long id_aa64mmfr0_el1;
  asm("mrs %0, S3_0_C0_C7_0" : "=r"(id_aa64mmfr0_el1));
  unsigned msa = (id_aa64mmfr0_el1 >> 48) & 0x0fU;
  unsigned msa_frac = (id_aa64mmfr0_el1 >> 52) & 0x0fU;

  // See Armv8-R AArch64 supplement (ARM DDI 0600A)
  if (pmsa && (msa == 0 || msa != 0xf || (msa_frac != 1 && msa_frac != 2)))
    L4Re::throw_error(-L4_ENOSYS, "CPU does not support PMSA");
  else if (!pmsa && !(msa == 0 || (msa == 0xf && msa_frac == 2)))
    L4Re::throw_error(-L4_ENOSYS, "CPU does not support VMSA");

  l4_vcpu_e_write_64(vcpu, L4_VCPU_E_VTCR, pmsa ? 0 : (1ULL << 31));

  // VPIDR default: Use MIDR_EL1 of this host CPU.
  l4_uint64_t midr_el1;
  asm volatile ("mrs %0, midr_el1" : "=r"(midr_el1));
  l4_vcpu_e_write_32(vcpu, L4_VCPU_E_VPIDR, midr_el1);
}

